// $Id: CExpGain.cpp,v 1.2 2007/03/03 02:59:53 paul Exp $

/*
 * All contents of this source code are copyright 2007 Exp Digital Uk.
 * This file forms part of the example projects for the Infinity API.
 * You may not redistribute either the source code or the compiled binary.
 * The Infinity API is covered by a fair use licence which you should have recieved with this
 * file. If you didnt receieve the licence, please contact us via http://www.expdigital.co.uk
 */

#include "CExpGain.hpp"
#include "CExpGainEditor.hpp"
using Exponent::ExpGain::CExpGain;
using Exponent::ExpGain::CExpGainEditor;

//	===========================================================================
/*
 * Make sure they include this macro at the top of your files to correctly initialise the run time type information
 */
EXPONENT_CLASS_IMPLEMENTATION(CExpGain, CVstEffect);

//	===========================================================================
CExpGain::CExpGain(audioMasterCallback audioMaster) 
		: CVstEffect(audioMaster, 1, 0)
		, m_gain(1.f)
{
	/*
	 * This increments the class creation count. NOTE that this is different from the reference counts
	 */
	EXPONENT_CLASS_CONSTRUCTION(CExpGain);

	/*
	 * We setup the standard VST stuff
	 */
	this->setNumInputs(2);
	this->setNumOutputs(2);
	this->canProcessReplacing();
	this->setUniqueID('ExEg');

	/*
	 * To create the editor, we create a set of attributes for the main window. 
	 * The attributes are created and we set a background colour. Generally you dont need to do
	 * anything more than this, and unless you are <absolutely sure> you understand what this changes
	 * For the most part just change the background colour
	 */
	CWindowAttributes attributes;
	// Set this to whatever you would like the background to be....
	attributes.setBackgroundColour(CColour::CCOLOUR_TURQUOISE);

	/*
	 * Now we construct the editor. This is destroyed by the VST base class, so we dont need to do anything with it after
	 * this point, besides parameter interaction
	 * The parameters expected are
	 * The effect (this)
	 * The size of the editor, what ever you want
	 * A pointer to the attributes you constructed above
	 */
	editor = new CExpGainEditor(this, CRect(0, 0, 120, 140), &attributes);
}

//	===========================================================================
CExpGain::~CExpGain()
{
	/*
	 * This decrements the class creation count
	 */
	EXPONENT_CLASS_DESTRUCTION(CExpGain);

	/*
	 * If we are on mac we need to stop the API from running
	 */
	#ifndef WIN32
		 STOP_EXPONENT_API_NO_LOG;
	#endif
}

//	===========================================================================
VstInt32 CExpGain::canDo(char *text)
{
	/*
	 * All plugins need to notify the host of what they are capable of
	 * Take a look at the VST 2.4 specs for more information, this isnt really part of
	 * Infinity API
	 */
	VstInt32 code = CVst::CVST_I_CANT_DO;
	if (text)
	{
		if		(!strcmp (text, "plugAsChannelInsert")) code = CVst::CVST_I_CAN_DO;
		else if (!strcmp (text, "plugAsSend"))			code = CVst::CVST_I_CAN_DO;
		else if (!strcmp (text, "2in2out"))				code = CVst::CVST_I_CAN_DO;
		else if (!strcmp (text, "receiveVstMidiEvent")) code = CVst::CVST_I_CAN_DO;
	}
	return code;
}

//	===========================================================================
bool CExpGain::getOutputProperties(VstInt32 index, VstPinProperties* properties)
{
	/*
	 * All plugins need to notify the host of what outputs they have
	 * Take a look at the VST 2.4 specs for more information, this isnt really part of
	 * Infinity API
	 */
	if (index == 0 || index == 1)
	{
		properties->flags = kVstPinIsActive | kVstPinIsStereo;
	}
	if (index == 0)
	{
		strcpy(properties->label,	   "Left output");
		strcpy(properties->shortLabel, "L out");
		return true;
	}
	else if (index == 1)
	{
		strcpy(properties->label,	   "Right output");
		strcpy(properties->shortLabel, "R out");
		return true;
	}
	return false;
}

//	===========================================================================
bool CExpGain::getInputProperties(VstInt32 index, VstPinProperties *properties)
{
	/*
	 * All plugins need to notify the host of what inputs they have
	 * Take a look at the VST 2.4 specs for more information, this isnt really part of
	 * Infinity API
	 */
	if (index == 0 || index == 1)
	{
		properties->flags = kVstPinIsActive | kVstPinIsStereo;
	}
	if (index == 0)
	{
		strcpy(properties->label,	   "Left input");
		strcpy(properties->shortLabel, "L in");
		return true;
	}
	else if (index == 1)
	{
		strcpy(properties->label,	   "Right input");
		strcpy(properties->shortLabel, "R in");
		return true;
	}
	return false;
}

//	===========================================================================
void CExpGain::process(float **input, float **output, VstInt32 numberOfSamples)
{
	/*
	 * This is the fun part, where the actual DSP takes place.
	 * There are plenty of places on the web that will give you algorithms that you can play with
	 * This function just changes the gain of the input - the amplitude of the output is scaled
	 * by the gain parameter
	 */
	for (long i = 0; i < numberOfSamples; i++)
	{
		// Process the left channel
		output[0][i] += input[0][i] * m_gain;

		// Process the right channel
		output[1][i] += input[1][i] * m_gain;
	}

	/*
	 * Make sure you call this to notify the effect that the process has finished
	 */
	CVstEffect::process(input, output, numberOfSamples);
}

//	===========================================================================
void CExpGain::processReplacing(float **input, float **output, VstInt32 numberOfSamples)
{
	/*
	 * This is the fun part, where the actual DSP takes place.
	 * There are plenty of places on the web that will give you algorithms that you can play with
	 * This function just changes the gain of the input - the amplitude of the output is scaled
	 * by the gain parameter
	 */
	for (long i = 0; i < numberOfSamples; i++)
	{
		// Process the left channel
		output[0][i] = input[0][i] * m_gain;

		// Process the right channel
		output[1][i] = input[1][i] * m_gain;
	}

	/*
	 * Make sure you call this to notify the effect that the process has finished
	 */
	CVstEffect::processReplacing(input, output, numberOfSamples);
}

//	===========================================================================
void CExpGain::setParameter(VstInt32 index, float value) 
{ 
	/*
	 * The host application has changed our parameter. We store the new parameter 
	 */
	m_gain = value;

	/*
	 * Now we need to notify the editor, but only if this message isnt from it!
	 */
	if (editor && !((CVstWindow *)editor)->parameterSetFromEditor())
	{
		((CVstWindow *)editor)->setParameter(index, m_gain);
	}
}

//	===========================================================================
float CExpGain::getParameter(VstInt32 index) 
{ 
	/*
	 * The host wants to know the value of our parameter
	 */
	return m_gain; 
}

//	===========================================================================
void CExpGain::getParameterLabel(VstInt32 index, char *label) 
{ 
	/*
	 * The host wants to know what the label of the parameter is
	 * This is labelled slightly oddly in the VSt spec. what they want is the 
	 * symbol, such as Hz, %, msec etc
	 */
	if (label)
	{
		CString string = "%";
		string.getString(label, CVst::CVST_STRING_LENGTH);
	}
}

//	===========================================================================
void CExpGain::getParameterDisplay(VstInt32 index, char *text) 
{ 
	/*
	 * The host wants to know what the label of the parameter is
	 * This is labelled slightly oddly in the VSt spec. what they want is the 
	 * warped value of the control, in this case a value in range 0 - 100
	 */
	if (text)
	{
		CString string;
		string.setStringWithFormat("%li", (long)(m_gain * 100.0));
		string.getString(text, CVst::CVST_STRING_LENGTH);
	}
}

//	===========================================================================
void CExpGain::getParameterName(VstInt32 index, char *text) 
{ 
	/*
	 * The host wants to know the name of the parameter
	 */
	if (text)
	{
		CString string = "The Gain";
		string.getString(text, CVst::CVST_STRING_LENGTH);
	}
}